using System;
using System.IO;

using Team_Project.Elements;

namespace Team_Project.PersistencyManagers.Protocols
{
	/// <summary>
	/// Interfaccia che dev'essere implementata da tutti i protocolli per l'accesso
	/// agli elementi di team project.
	/// <note>Se il protocollo prevede la presenza di pi copie  necessario implementare
	/// l'interfaccia specializzata
	/// <see cref="Team_Project.PersistencyManagers.Protocols.IDistribuitedPersistencyProtocol">
	/// IDistribuitedPersistencyProtocol</see></note>
	/// </summary>
	public interface IPersistencyProtocol
	{
		/// <summary>
		/// Nome del protocollo. Utilizzato per scopi di tracing.
		/// </summary>
		string Name
		{
			get;
		}

		/// <summary>
		/// Inizializza il protocollo
		/// </summary>
		void Intit();
		/// <summary>
		/// Rilascia eventuali risorse allocate dalla Init()
		/// </summary>
		void Close();

		/// <summary>
		/// Scrive un elemento nello storage opportuno
		/// </summary>
		/// <param name="el">Elemento da scrivere</param>
		void Store(IProjectElement el);
		/// <summary>
		/// Recupera dall'opportuno storage l'elemento desiderato
		/// </summary>
		/// <exception cref="Team_Project.Exceptions.ElementNotExistsException">L'elemento non esiste</exception>
		/// <exception cref="Team_Project.Exceptions.LocationNotExistsException">La location non esiste</exception>
		/// <param name="location">Location dell'elemento</param>
		/// <param name="name">Nome del'elemento</param>
		/// <param name="elementType">Tipo dell'elemento desiderato</param>
		/// <returns>L'elemento richiesto</returns>
		IProjectElement Retrive(string location,string name, Type elementType);
		/// <summary>
		/// Controlla se una location esiste
		/// </summary>
		/// <param name="location">Percorso completo della location da controllare</param>
		/// <returns>true se la location esiste, false altrimenti</returns>
		bool QueryLocationExists(string location);
		/// <summary>
		/// Controlla se un elemento esiste
		/// </summary>
		/// <param name="location">Percorso completo della location dove dovrebbe
		/// risiedere l'elemento</param>
		/// <param name="name">Nome dell'elemento</param>
		/// <returns>true se l'elemento esiste, false altrimenti</returns>
		bool QueryElementExists(string location,string name);
		/// <summary>
		/// Crea una location con il percorso specificato
		/// </summary>
		/// <param name="location">Percorso della location da creare</param>
		void CreateLocation(string location);
		/// <summary>
		/// Distrugge una location
		/// </summary>
		/// <param name="location">Percorso completo della location da distruggere</param>
		/// <param name="checkEmpty">Se true e la location che si cerca di distruggere
		/// non  vuota la funzione lancer un'eccezione. Se false il contenuto
		/// della location verr a sua volta distrutto</param>
		void DestroyLocation(string location, bool checkEmpty);
		/// <summary>
		/// Elimina un elemento
		/// </summary>
		/// <param name="el">Elemento da eliminare</param>
		void DestroyElement(IProjectElement el);
		/// <summary>
		/// Controlla quali elementi sono contenuti in una certa location
		/// </summary>
		/// <param name="location">Location di cui ottenere il contenuto</param>
		/// <returns>Un array con i nomi di tutti gli elementi contenuti
		/// in quella location</returns>
		string[] QueryElemensInLocation(string location);
		/// <summary>
		/// Controlla le sotto-location di primo livello rispetto ad una base
		/// </summary>
		/// <param name="baseL">Percorso della location di cui ottenere le 
		/// figlie dirette</param>
		/// <returns>Un array di stringhe contenente tutti i nomi delle sotto
		/// location di primo livello</returns>
		string[] QueryLocations(string baseL);
	}

	/// <summary>
	/// Questa interfaccia estende quella locale aggiungendo un parametro alle
	/// funzioni che richiedono scrittura e cancellazione. Questo parametro viene
	/// utilizzato per inoltrare le richieste alle altre copie senza re-inviare
	/// la richiesta alla sorgente.
	/// </summary>
	public interface IDistribuitedPersistencyProtocol : IPersistencyProtocol
	{
		/// <summary>
		/// <see cref="IPersistencyProtocol.Store">IPersistencyProtocol.Store</see>
		/// </summary>
		/// <param name="el"></param>
		/// <param name="source">Sorgente della richiesta</param>
		void Store(IProjectElement el,string source);
		/// <summary>
		/// <see cref="IPersistencyProtocol.CreateLocation">IPersistencyProtocol.CreateLocation</see>
		/// </summary>
		/// <param name="location"></param>
		/// <param name="source">Sorgente della richiesta</param>
		void CreateLocation(string location,string source);
		/// <summary>
		/// <see cref="IPersistencyProtocol.DestroyLocation">IPersistencyProtocol.DestroyLocation</see>
		/// </summary>
		/// <param name="location"></param>
		/// <param name="checkEmpty"></param>
		/// <param name="source">Sorgente della richiesta</param>
		void DestroyLocation(string location, bool checkEmpty,string source);
		/// <summary>
		/// <see cref="IPersistencyProtocol.DestroyElement">IPersistencyProtocol.DestroyElement</see>
		/// </summary>
		/// <param name="el"></param>
		/// <param name="source">Sorgente della richiesta</param>
		void DestroyElement(IProjectElement el,string source);
	}

	/// <summary>
	/// Delegato utilizzato per la richiesta asincrona di un'operazione di store
	/// </summary>
	public delegate void StoreDelegate(IProjectElement el,string source);
	/// <summary>
	/// Delegato utilizzato per la richiesta asincrona di un'operazione creazione
	/// di una location
	/// </summary>
	public delegate void CreateLocationDelegate(string location,string source);
	/// <summary>
	/// Delegato utilizzato per la richiesta asincrona di un'operazione distruzione
	/// di una location
	/// </summary>
	public delegate void DestroyLocationDelegate(string location, bool checkEmpty,string source);
	/// <summary>
	/// Delegato utilizzato per la richiesta asincrona di un'operazione creazione
	/// di un elemento
	/// </summary>
	public delegate void DestroyElementDelegate(IProjectElement el,string source);
}
